define([
    'jquery',
    'Magento_Checkout/js/view/payment/default',
    'Magento_Checkout/js/model/quote',
    'Magento_Checkout/js/action/place-order',
    'Magento_Checkout/js/model/full-screen-loader',
    'Magento_Customer/js/customer-data',
    'mage/url',
    'mage/storage',
    'ko'
], function ($, Component, quote, placeOrderAction, fullScreenLoader, customerData, urlBuilder, storage, ko) {
    'use strict';

    // Contract ABI for LayerPay
    var CONTRACT_ABI = [
        {
            "inputs": [
                {"name": "orderId", "type": "uint256"},
                {"name": "merchant", "type": "address"}
            ],
            "name": "pay",
            "outputs": [],
            "stateMutability": "payable",
            "type": "function"
        },
        {
            "inputs": [
                {"name": "orderId", "type": "uint256"},
                {"name": "merchant", "type": "address"},
                {"name": "token", "type": "address"},
                {"name": "amount", "type": "uint256"}
            ],
            "name": "payWithToken",
            "outputs": [],
            "stateMutability": "nonpayable",
            "type": "function"
        }
    ];

    return Component.extend({
        defaults: {
            template: 'LayerPay_CryptoPay/payment/layerpay',
            walletAddress: ko.observable(''),
            selectedCurrency: ko.observable('ETH'),
            cryptoAmount: ko.observable('0'),
            isWalletConnected: ko.observable(false),
            isProcessing: ko.observable(false),
            errorMessage: ko.observable(''),
            config: null,
            walletProvider: null
        },

        initialize: function () {
            var self = this;
            this._super();

            // DON'T clean sessions on load - breaks existing connections
            // cleanupStaleSessions is only called on explicit disconnect

            // Delay config load slightly to ensure component is fully ready
            setTimeout(function() {
                self.loadConfig();
            }, 100);

            this.initWeb3Modal();
            return this;
        },

        /**
         * Clean stale WalletConnect/Web3Modal sessions from localStorage
         */
        cleanupStaleSessions: function () {
            try {
                Object.keys(localStorage).forEach(function(key) {
                    if (key.startsWith('wc@') || key.startsWith('@w3m') || key.startsWith('W3M')) {
                        localStorage.removeItem(key);
                    }
                });
            } catch (e) {
                // Ignore errors
            }
        },

        /**
         * Initialize Web3Modal - matches PrestaShop exactly
         */
        initWeb3Modal: function () {
            var self = this;

            var checkAndSubscribe = function() {
                if (!window.web3Modal) {
                    setTimeout(checkAndSubscribe, 500);
                    return;
                }

                // Check existing connection (like PrestaShop)
                try {
                    var address = window.web3Modal.getAddress();
                    var provider = window.web3Modal.getWalletProvider();
                    if (address) {
                        self.walletAddress(address);
                        self.isWalletConnected(true);
                        self.walletProvider = provider;
                    }
                } catch (e) {
                    // Ignore errors
                }

                // Subscribe only once globally (like PrestaShop)
                if (!window._layerpaySubscribed) {
                    window._layerpaySubscribed = true;
                    window.web3Modal.subscribeProvider(function(state) {
                        if (state.isConnected && state.address && state.provider) {
                            self.walletAddress(state.address);
                            self.isWalletConnected(true);
                            self.walletProvider = state.provider;
                        } else if (!state.isConnected) {
                            self.walletAddress('');
                            self.isWalletConnected(false);
                            self.walletProvider = null;
                        }
                    });
                }
            };

            // Wait for Web3Modal to be ready
            if (window.web3ModalReady) {
                checkAndSubscribe();
            } else {
                window.addEventListener('web3ModalReady', function(e) {
                    if (e.detail && e.detail.modal) {
                        window.web3Modal = e.detail.modal;
                        window.web3ModalReady = true;
                    }
                    checkAndSubscribe();
                }, { once: true });
                // Delayed check as fallback
                setTimeout(checkAndSubscribe, 2000);
            }
        },

        loadConfig: function () {
            var self = this;
            storage.get(urlBuilder.build('rest/V1/layerpay/config'))
                .done(function (response) {
                    // Transform array response to object
                    self.config = {
                        enabled: response[0],
                        testMode: response[1],
                        network: response[2],
                        networkName: response[3],
                        chainId: response[4],
                        rpcUrl: response[5],
                        explorer: response[6],
                        merchantAddress: response[7],
                        contractAddress: response[8],
                        usdcAddress: response[9],
                        acceptedCurrencies: response[10]
                    };
                    self.updatePrice();
                })
                .fail(function () {
                    self.errorMessage('Failed to load payment configuration');
                });
        },

        getTitle: function () {
            return 'LayerPay (ETH & USDC)';
        },

        getCode: function () {
            return 'layerpay_cryptopay';
        },

        isActive: function () {
            return true;
        },

        getAcceptedCurrencies: function () {
            return this.config ? this.config.acceptedCurrencies : ['ETH', 'USDC'];
        },

        selectCurrency: function (currency) {
            this.selectedCurrency(currency);
            this.updatePrice();
        },

        updatePrice: function () {
            var self = this;
            var totals = quote.totals();

            if (!totals || !totals.grand_total) {
                setTimeout(function() { self.updatePrice(); }, 500);
                return;
            }

            var total = totals.grand_total;
            var currency = this.selectedCurrency();

            storage.get(urlBuilder.build('rest/V1/layerpay/price/' + currency + '/' + total))
                .done(function (response) {
                    // Response is array: [success, currency, fiatAmount, cryptoAmount, priceEur, priceUsd]
                    if (response[0] === true) {
                        self.cryptoAmount(response[3]);
                    }
                })
                .fail(function(error) {
                    // Silent fail for price fetch
                });
        },

        /**
         * Connect wallet - opens Web3Modal
         */
        connectWallet: function () {
            var self = this;
            self.errorMessage('');

            if (!window.web3Modal) {
                self.errorMessage('Web3Modal not ready. Please wait and try again.');
                return;
            }

            if (self.isWalletConnected()) {
                // Already connected - open account view to disconnect
                try {
                    if (typeof window.web3Modal.disconnect === 'function') {
                        window.web3Modal.disconnect();
                    } else {
                        window.web3Modal.open({ view: 'Account' });
                    }
                } catch (e) {
                    window.web3Modal.open({ view: 'Account' });
                }
                return;
            }

            // Open Web3Modal (shows MetaMask, WalletConnect, etc.)
            window.web3Modal.open();
        },

        switchNetwork: async function () {
            var self = this;
            if (!self.config || !self.walletProvider) return;

            var provider = self.walletProvider;
            var targetChainId = '0x' + self.config.chainId.toString(16);
            var currentChainId = await provider.request({ method: 'eth_chainId' });

            if (currentChainId !== targetChainId) {
                try {
                    await provider.request({
                        method: 'wallet_switchEthereumChain',
                        params: [{ chainId: targetChainId }]
                    });
                } catch (switchError) {
                    if (switchError.code === 4902) {
                        // Chain not added, try to add it
                        await provider.request({
                            method: 'wallet_addEthereumChain',
                            params: [{
                                chainId: targetChainId,
                                chainName: self.config.networkName,
                                rpcUrls: [self.config.rpcUrl],
                                blockExplorerUrls: [self.config.explorer]
                            }]
                        });
                    } else {
                        throw switchError;
                    }
                }
            }
        },

        disconnectWallet: function () {
            this.walletAddress('');
            this.isWalletConnected(false);
            this.walletProvider = null;

            // Clean all WalletConnect/Web3Modal sessions
            this.cleanupStaleSessions();
        },

        formatAddress: function (address) {
            if (!address) return '';
            return address.substring(0, 6) + '...' + address.substring(address.length - 4);
        },

        processPayment: async function () {
            var self = this;

            // EXACT COPY OF PRESTASHOP APPROACH
            // Check connection - also check Web3Modal state as fallback
            if (!self.isWalletConnected() && !self.walletAddress()) {
                if (window.web3Modal) {
                    var address = window.web3Modal.getAddress();
                    var provider = window.web3Modal.getWalletProvider();
                    if (address && provider) {
                        self.walletAddress(address);
                        self.walletProvider = provider;
                        self.isWalletConnected(true);
                    }
                }
            }

            if (!self.isWalletConnected() || !self.walletAddress()) {
                self.errorMessage('Please connect your wallet first');
                return;
            }

            // Get fresh provider from Web3Modal (like PrestaShop)
            var provider = self.walletProvider;
            if (window.web3Modal) {
                var wcProvider = window.web3Modal.getWalletProvider();
                if (wcProvider) {
                    provider = wcProvider;
                    self.walletProvider = wcProvider;
                }
            }

            if (!provider) {
                self.errorMessage('No wallet provider found');
                return;
            }

            // Verify provider is valid by requesting accounts
            // This ensures connection is still active after disconnect/reconnect
            try {
                var accounts = await provider.request({ method: 'eth_requestAccounts' });
                if (!accounts || accounts.length === 0) {
                    self.errorMessage('Wallet not connected. Please reconnect.');
                    self.isWalletConnected(false);
                    return;
                }
                // Update address in case it changed
                self.walletAddress(accounts[0]);
            } catch (e) {
                self.errorMessage('Wallet connection lost. Please reconnect.');
                self.isWalletConnected(false);
                return;
            }

            // Check network (like PrestaShop)
            var expectedChainId = '0x' + self.config.chainId.toString(16);
            try {
                var currentChainId = await provider.request({ method: 'eth_chainId' });
                if (currentChainId.toLowerCase() !== expectedChainId.toLowerCase()) {
                    await self.switchNetwork();
                    await self.sleep(1000);
                    currentChainId = await provider.request({ method: 'eth_chainId' });
                    if (currentChainId.toLowerCase() !== expectedChainId.toLowerCase()) {
                        self.errorMessage('Wrong network! Please switch to ' + self.config.networkName);
                        return;
                    }
                }
            } catch (e) {
                self.errorMessage('Please switch to ' + self.config.networkName + ' network');
                return;
            }

            self.isProcessing(true);
            self.errorMessage('');
            fullScreenLoader.startLoader();

            try {
                var currency = self.selectedCurrency();
                var amount = self.cryptoAmount();

                var txHash;
                if (currency === 'ETH') {
                    txHash = await self.sendETHPayment(amount);
                } else {
                    txHash = await self.sendTokenPayment(currency, amount);
                }

                var paymentData = {
                    method: self.getCode(),
                    additional_data: {
                        tx_hash: txHash,
                        wallet_address: self.walletAddress(),
                        crypto_currency: currency,
                        crypto_amount: amount
                    }
                };

                await self.placeOrder(paymentData);
                fullScreenLoader.stopLoader();
                self.isProcessing(false);

            } catch (error) {
                fullScreenLoader.stopLoader();
                self.isProcessing(false);
                var msg = error.message || 'Payment failed';
                if (error.code === 4001) {
                    msg = 'Transaction rejected by user';
                }
                self.errorMessage(msg);
            }
        },

        sendETHPayment: async function (amount) {
            var self = this;
            var contractAddress = self.config.contractAddress;

            if (!contractAddress) {
                throw new Error('Payment contract not configured');
            }

            // Convert amount to wei
            var amountWei = self.toWei(amount, 18);

            // All payments go through smart contract
            return await self.sendContractPayment(amountWei);
        },

        sendTokenPayment: async function (currency, amount) {
            var self = this;
            var contractAddress = self.config.contractAddress;
            var merchantAddress = self.config.merchantAddress;
            var tokenAddress = self.config.usdcAddress;

            if (!contractAddress) {
                throw new Error('Payment contract not configured');
            }

            if (!tokenAddress) {
                throw new Error('USDC not supported on this network');
            }

            // USDC has 6 decimals
            var decimals = 6;
            var amountUnits = self.toWei(amount, decimals);

            // Step 1: Approve contract to spend tokens
            var approveData = self.encodeApprove(contractAddress, amountUnits);
            var approveTxParams = {
                from: self.walletAddress(),
                to: tokenAddress,
                data: approveData
            };

            var approveTxHash = await self.walletProvider.request({
                method: 'eth_sendTransaction',
                params: [approveTxParams]
            });

            await self.waitForTransaction(approveTxHash);

            // Use quote ID like PrestaShop uses cartId
            var orderId = parseInt(quote.getQuoteId()) || Date.now() % 1000000;

            // Step 2: Call contract payWithToken(uint256 orderId, address merchant, address token, uint256 amount)
            var payWithTokenData = self.encodePayWithTokenFunction(orderId, merchantAddress, tokenAddress, amountUnits);

            var txParams = {
                from: self.walletAddress(),
                to: contractAddress,
                data: payWithTokenData
            };

            var txHash = await self.walletProvider.request({
                method: 'eth_sendTransaction',
                params: [txParams]
            });

            await self.waitForTransaction(txHash);
            return txHash;
        },

        sendContractPayment: async function (amountWei) {
            var self = this;
            var contractAddress = self.config.contractAddress;
            var merchantAddress = self.config.merchantAddress;

            // Use quote ID like PrestaShop uses cartId
            var orderId = parseInt(quote.getQuoteId()) || Date.now() % 1000000;

            // Encode pay(uint256 orderId, address merchant) - amount comes from msg.value
            var payData = self.encodePayFunction(orderId, merchantAddress);

            var txParams = {
                from: self.walletAddress(),
                to: contractAddress,
                value: amountWei,
                data: payData
            };

            var txHash = await self.walletProvider.request({
                method: 'eth_sendTransaction',
                params: [txParams]
            });

            await self.waitForTransaction(txHash);
            return txHash;
        },

        waitForTransaction: async function (txHash) {
            var self = this;
            var maxAttempts = 60;
            var attempts = 0;

            while (attempts < maxAttempts) {
                var receipt = await self.walletProvider.request({
                    method: 'eth_getTransactionReceipt',
                    params: [txHash]
                });

                if (receipt) {
                    if (receipt.status === '0x1') {
                        return receipt;
                    } else {
                        throw new Error('Transaction failed');
                    }
                }

                await self.sleep(2000);
                attempts++;
            }

            throw new Error('Transaction confirmation timeout');
        },

        sleep: function (ms) {
            return new Promise(resolve => setTimeout(resolve, ms));
        },

        toWei: function (amount, decimals) {
            var factor = Math.pow(10, decimals);
            var wei = Math.floor(parseFloat(amount) * factor);
            return '0x' + wei.toString(16);
        },

        encodeApprove: function (spender, amount) {
            // ERC20 approve(address,uint256) selector: 0x095ea7b3
            var selector = '095ea7b3';
            var paddedSpender = spender.toLowerCase().replace('0x', '').padStart(64, '0');
            var paddedAmount = amount.replace('0x', '').padStart(64, '0');
            return '0x' + selector + paddedSpender + paddedAmount;
        },

        encodePayFunction: function (orderId, merchant) {
            // pay(uint256,address) selector: 0x31cbf5e3
            var selector = '31cbf5e3';
            var paddedOrderId = orderId.toString(16).padStart(64, '0');
            var paddedMerchant = merchant.toLowerCase().replace('0x', '').padStart(64, '0');
            return '0x' + selector + paddedOrderId + paddedMerchant;
        },

        encodePayWithTokenFunction: function (orderId, merchant, token, amount) {
            // payWithToken(uint256,address,address,uint256) selector: 0xd6bcaa76
            var selector = 'd6bcaa76';
            var paddedOrderId = orderId.toString(16).padStart(64, '0');
            var paddedMerchant = merchant.toLowerCase().replace('0x', '').padStart(64, '0');
            var paddedToken = token.toLowerCase().replace('0x', '').padStart(64, '0');
            var paddedAmount = amount.replace('0x', '').padStart(64, '0');
            return '0x' + selector + paddedOrderId + paddedMerchant + paddedToken + paddedAmount;
        },

        stringToBytes32: function (str) {
            var hex = '';
            for (var i = 0; i < str.length && i < 32; i++) {
                hex += str.charCodeAt(i).toString(16).padStart(2, '0');
            }
            return '0x' + hex.padEnd(64, '0');
        },

        placeOrder: function (paymentData) {
            var self = this;

            return placeOrderAction(paymentData, false, self.messageContainer)
                .done(function () {
                    customerData.invalidate(['cart']);
                    $.mage.redirect(urlBuilder.build('checkout/onepage/success'));
                });
        },

        getData: function () {
            return {
                method: this.getCode(),
                additional_data: {
                    tx_hash: '',
                    wallet_address: this.walletAddress(),
                    crypto_currency: this.selectedCurrency(),
                    crypto_amount: this.cryptoAmount()
                }
            };
        }
    });
});
