<?php
/**
 * LayerPay CryptoPay
 *
 * @copyright Copyright (c) 2024 LayerPay
 * @license   Proprietary
 */
declare(strict_types=1);

namespace LayerPay\CryptoPay\Model\Payment;

use Magento\Payment\Model\Method\AbstractMethod;
use Magento\Framework\Model\Context;
use Magento\Framework\Registry;
use Magento\Framework\Api\ExtensionAttributesFactory;
use Magento\Framework\Api\AttributeValueFactory;
use Magento\Payment\Helper\Data;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Payment\Model\Method\Logger;
use Magento\Framework\Model\ResourceModel\AbstractResource;
use Magento\Framework\Data\Collection\AbstractDb;

class CryptoPay extends AbstractMethod
{
    public const CODE = 'layerpay_cryptopay';

    protected $_code = self::CODE;
    protected $_isGateway = true;
    protected $_canAuthorize = true;
    protected $_canCapture = true;
    protected $_canCapturePartial = false;
    protected $_canRefund = false;
    protected $_canVoid = false;
    protected $_canUseInternal = false;
    protected $_canUseCheckout = true;
    protected $_canUseForMultishipping = false;
    protected $_isInitializeNeeded = false;

    public function __construct(
        Context $context,
        Registry $registry,
        ExtensionAttributesFactory $extensionFactory,
        AttributeValueFactory $customAttributeFactory,
        Data $paymentData,
        ScopeConfigInterface $scopeConfig,
        Logger $logger,
        AbstractResource $resource = null,
        AbstractDb $resourceCollection = null,
        array $data = []
    ) {
        parent::__construct(
            $context,
            $registry,
            $extensionFactory,
            $customAttributeFactory,
            $paymentData,
            $scopeConfig,
            $logger,
            $resource,
            $resourceCollection,
            $data
        );
    }

    /**
     * Get payment method title
     */
    public function getTitle()
    {
        return $this->getConfigData('title') ?: 'LayerPay (ETH & USDC)';
    }

    /**
     * Check if payment method is available
     */
    public function isAvailable(\Magento\Quote\Api\Data\CartInterface $quote = null)
    {
        if (!parent::isAvailable($quote)) {
            return false;
        }

        $merchantAddress = $this->getConfigData('merchant_address');
        if (empty($merchantAddress)) {
            return false;
        }

        return true;
    }

    /**
     * Authorize payment
     */
    public function authorize(\Magento\Payment\Model\InfoInterface $payment, $amount)
    {
        $payment->setTransactionId($payment->getAdditionalInformation('tx_hash'));
        $payment->setIsTransactionClosed(false);
        return $this;
    }

    /**
     * Capture payment
     */
    public function capture(\Magento\Payment\Model\InfoInterface $payment, $amount)
    {
        $txHash = $payment->getAdditionalInformation('tx_hash');
        if ($txHash) {
            $payment->setTransactionId($txHash . '-capture');
            $payment->setParentTransactionId($txHash);
        }
        return $this;
    }
}
