<?php
/**
 * LayerPay CryptoPay
 *
 * @copyright Copyright (c) 2024 LayerPay
 * @license   Proprietary
 */
declare(strict_types=1);

namespace LayerPay\CryptoPay\Model;

use LayerPay\CryptoPay\Api\ConfigInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Store\Model\ScopeInterface;

class Config implements ConfigInterface
{
    private const XML_PATH_ACTIVE = 'payment/layerpay_cryptopay/active';
    private const XML_PATH_TESTMODE = 'payment/layerpay_cryptopay/testmode';
    private const XML_PATH_NETWORK = 'payment/layerpay_cryptopay/network';
    private const XML_PATH_MERCHANT_ADDRESS = 'payment/layerpay_cryptopay/merchant_address';
    private const XML_PATH_ACCEPTED_CURRENCIES = 'payment/layerpay_cryptopay/accepted_currencies';

    private const NETWORKS = [
        // Testnets
        'sepolia' => [
            'name' => 'Sepolia Testnet',
            'chainId' => 11155111,
            'rpcUrl' => 'https://rpc.sepolia.org',
            'explorer' => 'https://sepolia.etherscan.io',
            'currency' => 'ETH',
            'isTestnet' => true,
            'contract' => '0x027811E894b6388C514f909d54921a701337f467',
            'usdc' => '0x7474e771f6f3d8123aa4cDD8d3593866651a14e6'
        ],
        'base_sepolia' => [
            'name' => 'Base Sepolia',
            'chainId' => 84532,
            'rpcUrl' => 'https://sepolia.base.org',
            'explorer' => 'https://sepolia.basescan.org',
            'currency' => 'ETH',
            'isTestnet' => true,
            'contract' => '0xF0DCC0C62587804d9c49B075d24725A9a6eA2c6E',
            'usdc' => '0x0f411ff500f88BB528b800C7116c28d80f8BbD44'
        ],
        'optimism_sepolia' => [
            'name' => 'Optimism Sepolia',
            'chainId' => 11155420,
            'rpcUrl' => 'https://sepolia.optimism.io',
            'explorer' => 'https://sepolia-optimism.etherscan.io',
            'currency' => 'ETH',
            'isTestnet' => true,
            'contract' => '0x3E9334D16A57ADC0cAb7Cea24703aC819c1DAB8D',
            'usdc' => '0x0f411ff500f88BB528b800C7116c28d80f8BbD44'
        ],
        'arbitrum_sepolia' => [
            'name' => 'Arbitrum Sepolia',
            'chainId' => 421614,
            'rpcUrl' => 'https://sepolia-rollup.arbitrum.io/rpc',
            'explorer' => 'https://sepolia.arbiscan.io',
            'currency' => 'ETH',
            'isTestnet' => true,
            'contract' => '0xC5913aE49d6C52267B58824297EC36d36c27740d',
            'usdc' => '0xd95480E52E671b87D6de3A3F05fbAb0E8526843F'
        ],
        // Mainnets
        'ethereum' => [
            'name' => 'Ethereum Mainnet',
            'chainId' => 1,
            'rpcUrl' => 'https://eth.llamarpc.com',
            'explorer' => 'https://etherscan.io',
            'currency' => 'ETH',
            'isTestnet' => false,
            'contract' => '0x84f679497947f9186258Af929De2e760677D5949',
            'usdc' => '0xA0b86991c6218b36c1d19D4a2e9Eb0cE3606eB48'
        ],
        'base' => [
            'name' => 'Base',
            'chainId' => 8453,
            'rpcUrl' => 'https://mainnet.base.org',
            'explorer' => 'https://basescan.org',
            'currency' => 'ETH',
            'isTestnet' => false,
            'contract' => '0x84f679497947f9186258Af929De2e760677D5949',
            'usdc' => '0x833589fCD6eDb6E08f4c7C32D4f71b54bdA02913'
        ],
        'optimism' => [
            'name' => 'Optimism',
            'chainId' => 10,
            'rpcUrl' => 'https://mainnet.optimism.io',
            'explorer' => 'https://optimistic.etherscan.io',
            'currency' => 'ETH',
            'isTestnet' => false,
            'contract' => '0x84f679497947f9186258Af929De2e760677D5949',
            'usdc' => '0x0b2C639c533813f4Aa9D7837CAf62653d097Ff85'
        ],
        'arbitrum' => [
            'name' => 'Arbitrum One',
            'chainId' => 42161,
            'rpcUrl' => 'https://arb1.arbitrum.io/rpc',
            'explorer' => 'https://arbiscan.io',
            'currency' => 'ETH',
            'isTestnet' => false,
            'contract' => '0x84f679497947f9186258Af929De2e760677D5949',
            'usdc' => '0xaf88d065e77c8cC2239327C5EDb3A432268e5831'
        ]
    ];

    private ScopeConfigInterface $scopeConfig;

    public function __construct(ScopeConfigInterface $scopeConfig)
    {
        $this->scopeConfig = $scopeConfig;
    }

    public function isEnabled(): bool
    {
        return $this->scopeConfig->isSetFlag(
            self::XML_PATH_ACTIVE,
            ScopeInterface::SCOPE_STORE
        );
    }

    public function isTestMode(): bool
    {
        return $this->scopeConfig->isSetFlag(
            self::XML_PATH_TESTMODE,
            ScopeInterface::SCOPE_STORE
        );
    }

    public function getNetwork(): string
    {
        return (string) $this->scopeConfig->getValue(
            self::XML_PATH_NETWORK,
            ScopeInterface::SCOPE_STORE
        ) ?: 'sepolia';
    }

    public function getMerchantAddress(): string
    {
        return (string) $this->scopeConfig->getValue(
            self::XML_PATH_MERCHANT_ADDRESS,
            ScopeInterface::SCOPE_STORE
        );
    }

    public function getContractAddress(): string
    {
        $networkConfig = $this->getNetworkConfig();
        return $networkConfig['contract'] ?? '';
    }

    public function getUsdcAddress(): string
    {
        $networkConfig = $this->getNetworkConfig();
        return $networkConfig['usdc'] ?? '';
    }

    public function getAcceptedCurrencies(): array
    {
        $currencies = $this->scopeConfig->getValue(
            self::XML_PATH_ACCEPTED_CURRENCIES,
            ScopeInterface::SCOPE_STORE
        );
        return $currencies ? explode(',', $currencies) : ['ETH', 'USDC'];
    }

    public function getNetworkConfig(): array
    {
        $network = $this->getNetwork();
        return self::NETWORKS[$network] ?? self::NETWORKS['sepolia'];
    }

    public function getRpcUrl(): string
    {
        return $this->getNetworkConfig()['rpcUrl'];
    }

    public function getChainId(): int
    {
        return $this->getNetworkConfig()['chainId'];
    }

    public function getAllNetworks(): array
    {
        return self::NETWORKS;
    }
}
