<?php
/**
 * LayerPay CryptoPay
 *
 * @copyright Copyright (c) 2024 LayerPay
 * @license   Proprietary
 */
declare(strict_types=1);

namespace LayerPay\CryptoPay\Model\Api;

use LayerPay\CryptoPay\Api\CryptoPayInterface;
use LayerPay\CryptoPay\Api\ConfigInterface;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Model\Order;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Psr\Log\LoggerInterface;

class CryptoPay implements CryptoPayInterface
{
    private ConfigInterface $config;
    private OrderRepositoryInterface $orderRepository;
    private SearchCriteriaBuilder $searchCriteriaBuilder;
    private LoggerInterface $logger;

    public function __construct(
        ConfigInterface $config,
        OrderRepositoryInterface $orderRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        LoggerInterface $logger
    ) {
        $this->config = $config;
        $this->orderRepository = $orderRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->logger = $logger;
    }

    public function getConfig(): array
    {
        $networkConfig = $this->config->getNetworkConfig();

        return [
            'enabled' => $this->config->isEnabled(),
            'testMode' => $this->config->isTestMode(),
            'network' => $this->config->getNetwork(),
            'networkName' => $networkConfig['name'],
            'chainId' => $networkConfig['chainId'],
            'rpcUrl' => $networkConfig['rpcUrl'],
            'explorer' => $networkConfig['explorer'],
            'merchantAddress' => $this->config->getMerchantAddress(),
            'contractAddress' => $this->config->getContractAddress(),
            'usdcAddress' => $this->config->getUsdcAddress(),
            'acceptedCurrencies' => $this->config->getAcceptedCurrencies()
        ];
    }

    public function getCryptoPrice(string $currency, float $amount): array
    {
        try {
            $currency = strtoupper($currency);

            // Get price from CoinGecko API
            $coinId = $this->getCoinGeckoId($currency);
            if (!$coinId) {
                return [
                    'success' => false,
                    'error' => 'Unsupported currency: ' . $currency
                ];
            }

            $url = "https://api.coingecko.com/api/v3/simple/price?ids={$coinId}&vs_currencies=eur,usd";

            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 10);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Accept: application/json'
            ]);

            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            if ($httpCode !== 200 || !$response) {
                throw new \Exception('Failed to fetch price from CoinGecko');
            }

            $data = json_decode($response, true);
            if (!isset($data[$coinId])) {
                throw new \Exception('Invalid response from CoinGecko');
            }

            $priceEur = $data[$coinId]['eur'] ?? 0;
            $priceUsd = $data[$coinId]['usd'] ?? 0;

            // USDC is 1:1 with USD
            if ($currency === 'USDC') {
                $cryptoAmount = $amount;
            } else {
                $cryptoAmount = $priceEur > 0 ? $amount / $priceEur : 0;
            }

            return [
                'success' => true,
                'currency' => $currency,
                'fiatAmount' => $amount,
                'cryptoAmount' => number_format($cryptoAmount, 8, '.', ''),
                'priceEur' => $priceEur,
                'priceUsd' => $priceUsd
            ];

        } catch (\Exception $e) {
            $this->logger->error('LayerPay price fetch error: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    public function verifyTransaction(string $txHash, string $orderId): array
    {
        try {
            $rpcUrl = $this->config->getRpcUrl();

            // Get transaction receipt via RPC
            $payload = json_encode([
                'jsonrpc' => '2.0',
                'method' => 'eth_getTransactionReceipt',
                'params' => [$txHash],
                'id' => 1
            ]);

            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $rpcUrl);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_TIMEOUT, 30);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'Content-Type: application/json'
            ]);

            $response = curl_exec($ch);
            curl_close($ch);

            if (!$response) {
                return [
                    'success' => false,
                    'verified' => false,
                    'error' => 'Failed to connect to RPC'
                ];
            }

            $data = json_decode($response, true);

            if (!isset($data['result']) || $data['result'] === null) {
                return [
                    'success' => true,
                    'verified' => false,
                    'status' => 'pending',
                    'message' => 'Transaction not yet confirmed'
                ];
            }

            $receipt = $data['result'];
            $status = hexdec($receipt['status'] ?? '0x0');

            return [
                'success' => true,
                'verified' => $status === 1,
                'status' => $status === 1 ? 'confirmed' : 'failed',
                'blockNumber' => hexdec($receipt['blockNumber'] ?? '0x0'),
                'gasUsed' => hexdec($receipt['gasUsed'] ?? '0x0')
            ];

        } catch (\Exception $e) {
            $this->logger->error('LayerPay verification error: ' . $e->getMessage());
            return [
                'success' => false,
                'verified' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    public function processPayment(
        string $orderId,
        string $txHash,
        string $walletAddress,
        string $currency,
        string $amount
    ): array {
        try {
            // Load order by increment ID
            $searchCriteria = $this->searchCriteriaBuilder
                ->addFilter('increment_id', $orderId)
                ->create();

            $orders = $this->orderRepository->getList($searchCriteria)->getItems();

            if (empty($orders)) {
                return [
                    'success' => false,
                    'error' => 'Order not found'
                ];
            }

            $order = reset($orders);

            // Verify transaction on blockchain
            $verification = $this->verifyTransaction($txHash, $orderId);

            if (!$verification['verified']) {
                return [
                    'success' => false,
                    'error' => 'Transaction not verified on blockchain',
                    'verification' => $verification
                ];
            }

            // Update order payment
            $payment = $order->getPayment();
            $payment->setAdditionalInformation('tx_hash', $txHash);
            $payment->setAdditionalInformation('wallet_address', $walletAddress);
            $payment->setAdditionalInformation('crypto_currency', $currency);
            $payment->setAdditionalInformation('crypto_amount', $amount);
            $payment->setTransactionId($txHash);

            // Set order status to processing
            $order->setState(Order::STATE_PROCESSING);
            $order->setStatus(Order::STATE_PROCESSING);
            $order->addCommentToStatusHistory(
                sprintf(
                    'LayerPay payment received. TX: %s, Amount: %s %s, Wallet: %s',
                    $txHash,
                    $amount,
                    $currency,
                    $walletAddress
                )
            );

            $this->orderRepository->save($order);

            $this->logger->info('LayerPay payment processed', [
                'order_id' => $orderId,
                'tx_hash' => $txHash,
                'currency' => $currency,
                'amount' => $amount
            ]);

            return [
                'success' => true,
                'orderId' => $orderId,
                'status' => 'processing',
                'message' => 'Payment processed successfully'
            ];

        } catch (\Exception $e) {
            $this->logger->error('LayerPay payment processing error: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    private function getCoinGeckoId(string $currency): ?string
    {
        $mapping = [
            'ETH' => 'ethereum',
            'USDC' => 'usd-coin',
            'MATIC' => 'matic-network',
            'USDT' => 'tether'
        ];

        return $mapping[$currency] ?? null;
    }
}
